import { cleanHTML } from './utils';

const DEFAULT_KEYS = [
    "",


];

const CONFIG = {
    model: "gemini-flash-latest", // Updated as per user request
    version: "v1beta",
    temperature: 0.7,
    maxOutputTokens: 64000, // Standard max for flash
    retryDelay: 1000
};

let currentKeyIndex = 0;

function getNextApiKey(keys = DEFAULT_KEYS) {
    if (!keys || keys.length === 0) return null;
    const key = keys[currentKeyIndex];
    currentKeyIndex = (currentKeyIndex + 1) % keys.length;
    return key;
}

export async function callAdvancedGeminiApi(prompt, systemPrompt, taskType, customKeys = null, onLog = () => { }, signal = null) {
    const keys = customKeys && customKeys.length > 0 ? customKeys : DEFAULT_KEYS;
    let attempts = 0;
    const maxAttempts = keys.length;

    while (attempts < maxAttempts) {
        // External Signal Abort Check
        if (signal?.aborted) {
            throw new Error("Aborted by user");
        }

        let apiKey = getNextApiKey(keys);
        onLog(`🔑 Using API key index ${currentKeyIndex} for ${taskType}`, 'info');

        const url = `https://generativelanguage.googleapis.com/${CONFIG.version}/models/${CONFIG.model}:generateContent?key=${apiKey}`;

        const payload = {
            systemInstruction: {
                parts: [{ text: systemPrompt || 'You are a world-class SEO strategist, copywriter, and knowledge systems engineer with 32 years of experience. You create long-form, Semantic google HCS , authoritative search-first, semantically optimized content. Every output must demonstrate topical authority, E-E-A-T signals, and deliver authentic, knowledge-rich' }]
            },
            contents: [{
                role: "user",
                parts: [{ text: prompt }]
            }],
            tools: [
                {
                    googleSearch: {}
                }
            ],
            generationConfig: {
                temperature: CONFIG.temperature,
                maxOutputTokens: CONFIG.maxOutputTokens,
                thinkingConfig: {
                    thinkingBudget: 2000
                }
            }
        };

        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 120000); // 120s Timeout

        // Connect external signal
        const onAbort = () => controller.abort();
        if (signal) signal.addEventListener('abort', onAbort);

        try {
            const response = await fetch(url, {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(payload),
                signal: controller.signal
            });

            clearTimeout(timeoutId);
            if (signal) signal.removeEventListener('abort', onAbort);

            const data = await response.json();

            if (!response.ok) {
                const errorMsg = data.error?.message || response.statusText;
                onLog(`API Error ${response.status}: ${errorMsg}`, 'error');

                if (response.status === 429) {
                    onLog(`Rate limited, trying next key...`, 'warning');
                    attempts++;
                    await new Promise(r => setTimeout(r, CONFIG.retryDelay));
                    continue;
                }
                attempts++;
                await new Promise(r => setTimeout(r, CONFIG.retryDelay));
                continue;
            }

            if (data.candidates && data.candidates[0].content) {
                onLog(`✅ Successfully generated ${taskType}`, 'success');
                return data.candidates[0].content.parts[0].text;
            } else {
                onLog(`Invalid response structure`, 'error');
                attempts++;
            }

        } catch (e) {
            clearTimeout(timeoutId);
            if (signal) signal.removeEventListener('abort', onAbort);

            if (e.name === 'AbortError') {
                if (signal?.aborted) throw new Error("Aborted by user");
                onLog(`Request timed out (120s)`, 'error');
                // If timeout, maybe retry? 
                // But typically timeout means network or model issue.
                // We count as attempt failure.
            } else {
                onLog(`Network/API Exception: ${e.toString()}`, 'error');
            }
            attempts++;
        }

        await new Promise(r => setTimeout(r, CONFIG.retryDelay));
    }

    onLog(`❌ All API keys exhausted for ${taskType}`, 'error');
    throw new Error("ALL_KEYS_EXHAUSTED");
}
